"""QGIS Unit tests for QgsDateTimeEdit

.. note:: This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
"""

__author__ = "Denis Rouzaud"
__date__ = "2018-01-04"
__copyright__ = "Copyright 2017, The QGIS Project"

from qgis.PyQt.QtCore import QDate, QDateTime, Qt, QTime, QT_VERSION_STR
from qgis.gui import QgsDateEdit, QgsDateTimeEdit, QgsTimeEdit
import unittest
from qgis.testing import start_app, QgisTestCase

start_app()

DATE = QDateTime.fromString("2018-01-01 01:02:03", Qt.DateFormat.ISODate)
DATE_Z = QDateTime.fromString("2018-01-01 01:02:03Z", Qt.DateFormat.ISODate)
DATE_OFFSET = QDateTime.fromString("2025-01-20T12:00:00+03:00", Qt.DateFormat.ISODate)


class TestQgsDateTimeEdit(QgisTestCase):

    def check_time_zone(self, widget, expected_date):
        if int(QT_VERSION_STR.split(".")[0]) > 6 or (
            int(QT_VERSION_STR.split(".")[0]) == 6
            and int(QT_VERSION_STR.split(".")[1]) >= 7
        ):
            self.assertEqual(widget.timeZone().id(), expected_date.timeZone().id())
        else:
            self.assertEqual(widget.timeSpec(), expected_date.timeSpec())

    def testSettersGetters(self):
        """test widget handling of null values"""
        for date in [DATE, DATE_Z, DATE_OFFSET]:
            w = QgsDateTimeEdit()
            w.setAllowNull(False)

            w.setDateTime(date)
            self.assertEqual(w.dateTime(), date)
            self.check_time_zone(w, date)
            # date should remain when setting an invalid date
            w.setDateTime(QDateTime())
            self.assertEqual(w.dateTime(), date)
            self.check_time_zone(w, date)

    def testNullValueHandling(self):
        """test widget handling of null values"""
        w = QgsDateTimeEdit()
        w.setAllowNull(True)

        # date should be valid again when not allowing NULL values
        w.setDateTime(QDateTime())
        w.setAllowNull(False)
        self.assertTrue(w.dateTime().isValid())

        w.setAllowNull(True)
        w.setDateTime(QDateTime())
        self.assertFalse(w.dateTime().isValid())

        w.setAllowNull(False)
        self.assertTrue(w.dateTime().isValid())


class TestQgsDateEdit(QgisTestCase):

    def testSettersGetters(self):
        """test widget handling of null values"""
        w = QgsDateEdit()
        w.setAllowNull(False)

        w.setDate(DATE.date())
        self.assertEqual(w.date(), DATE.date())
        # date should remain when setting an invalid date
        w.setDate(QDate())
        self.assertEqual(w.date(), DATE.date())

    def testNullValueHandling(self):
        """test widget handling of null values"""
        w = QgsDateEdit()
        w.setAllowNull(True)

        # date should be valid again when not allowing NULL values
        w.setDate(QDate())
        w.setAllowNull(False)
        self.assertTrue(w.date().isValid())

        w.setAllowNull(True)
        w.setDate(QDate())
        self.assertFalse(w.date().isValid())

        w.setAllowNull(False)
        self.assertTrue(w.date().isValid())


class TestQgsTimeEdit(QgisTestCase):

    def testSettersGetters(self):
        """test widget handling of null values"""
        w = QgsTimeEdit()
        w.setAllowNull(False)

        w.setTime(DATE.time())
        self.assertEqual(w.time(), DATE.time())
        # time should remain when setting an invalid time
        w.setTime(QTime())
        self.assertEqual(w.time(), DATE.time())

    def testNullValueHandling(self):
        """test widget handling of null values"""
        w = QgsTimeEdit()
        w.setAllowNull(True)

        # time should be valid again when not allowing NULL values
        w.setTime(QTime())
        w.setAllowNull(False)
        self.assertTrue(w.time().isValid())

        w.setAllowNull(True)
        w.setTime(QTime())
        self.assertFalse(w.time().isValid())

        w.setAllowNull(False)
        self.assertTrue(w.time().isValid())


if __name__ == "__main__":
    unittest.main()
