/***************************************************************************
  qgsfieldmappingwidget.cpp - QgsFieldMappingWidget

 ---------------------
 begin                : 16.3.2020
 copyright            : (C) 2020 by Alessandro Pasotti
 email                : elpaso at itopen dot it
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qgsfieldmappingwidget.h"
#include "moc_qgsfieldmappingwidget.cpp"
#include "qgsfieldexpressionwidget.h"
#include "qgsexpression.h"
#include "qgsprocessingaggregatewidgets.h"
#include "qgsvectorlayer.h"
#include "qgsvectordataprovider.h"
#include "QItemSelectionModel"

#include <QTableView>
#include <QVBoxLayout>

#ifdef ENABLE_MODELTEST
#include "modeltest.h"
#endif

QgsFieldMappingWidget::QgsFieldMappingWidget(
  QWidget *parent,
  const QgsFields &sourceFields,
  const QgsFields &destinationFields,
  const QMap<QString, QString> &expressions,
  const QList< QgsVectorDataProvider::NativeType > &nativeTypes
)
  : QgsPanelWidget( parent )
{
  QVBoxLayout *verticalLayout = new QVBoxLayout();
  verticalLayout->setContentsMargins( 0, 0, 0, 0 );
  mTableView = new QTableView();
  verticalLayout->addWidget( mTableView );
  setLayout( verticalLayout );

  mModel = new QgsFieldMappingModel( sourceFields, destinationFields, expressions, this );
  mModel->setNativeTypes( nativeTypes );

#ifdef ENABLE_MODELTEST
  new ModelTest( mModel, this );
#endif

  mTableView->setModel( mModel );
  mTableView->setItemDelegateForColumn( static_cast<int>( QgsFieldMappingModel::ColumnDataIndex::SourceExpression ), new QgsFieldMappingExpressionDelegate( this ) );
  mTypeDelegate = new QgsFieldMappingTypeDelegate( nativeTypes, mTableView );
  mTableView->setItemDelegateForColumn( static_cast<int>( QgsFieldMappingModel::ColumnDataIndex::DestinationType ), mTypeDelegate );
  updateColumns();
  // Make sure columns are updated when rows are added
  connect( mModel, &QgsFieldMappingModel::rowsInserted, this, [this] { updateColumns(); } );
  connect( mModel, &QgsFieldMappingModel::modelReset, this, [this] { updateColumns(); } );
  connect( mModel, &QgsFieldMappingModel::dataChanged, this, &QgsFieldMappingWidget::changed );
  connect( mModel, &QgsFieldMappingModel::rowsInserted, this, &QgsFieldMappingWidget::changed );
  connect( mModel, &QgsFieldMappingModel::rowsRemoved, this, &QgsFieldMappingWidget::changed );
  connect( mModel, &QgsFieldMappingModel::modelReset, this, &QgsFieldMappingWidget::changed );
}

void QgsFieldMappingWidget::setDestinationEditable( bool editable )
{
  qobject_cast<QgsFieldMappingModel *>( mModel )->setDestinationEditable( editable );
  updateColumns();
}

bool QgsFieldMappingWidget::destinationEditable() const
{
  return qobject_cast<QgsFieldMappingModel *>( mModel )->destinationEditable();
}

QgsFieldMappingModel *QgsFieldMappingWidget::model() const
{
  return qobject_cast<QgsFieldMappingModel *>( mModel );
}

QList<QgsFieldMappingModel::Field> QgsFieldMappingWidget::mapping() const
{
  return model()->mapping();
}

QMap<QString, QgsProperty> QgsFieldMappingWidget::fieldPropertyMap() const
{
  return model()->fieldPropertyMap();
}

void QgsFieldMappingWidget::setFieldPropertyMap( const QMap<QString, QgsProperty> &map )
{
  model()->setFieldPropertyMap( map );
}

QItemSelectionModel *QgsFieldMappingWidget::selectionModel()
{
  return mTableView->selectionModel();
}

void QgsFieldMappingWidget::setSourceFields( const QgsFields &sourceFields )
{
  model()->setSourceFields( sourceFields );
}

void QgsFieldMappingWidget::setSourceLayer( QgsVectorLayer *layer )
{
  mSourceLayer = layer;
}

QgsVectorLayer *QgsFieldMappingWidget::sourceLayer()
{
  return mSourceLayer;
}

void QgsFieldMappingWidget::setDestinationFields( const QgsFields &destinationFields, const QMap<QString, QString> &expressions )
{
  model()->setDestinationFields( destinationFields, expressions );
}

void QgsFieldMappingWidget::setNativeTypes( const QList<QgsVectorDataProvider::NativeType> &nativeTypes )
{
  mTypeDelegate->setNativeTypes( nativeTypes );
  mModel->setNativeTypes( nativeTypes );
}

void QgsFieldMappingWidget::scrollTo( const QModelIndex &index ) const
{
  mTableView->scrollTo( index );
}

void QgsFieldMappingWidget::registerExpressionContextGenerator( const QgsExpressionContextGenerator *generator )
{
  model()->setBaseExpressionContextGenerator( generator );
}

void QgsFieldMappingWidget::appendField( const QgsField &field, const QString &expression )
{
  model()->appendField( field, expression );
}

bool QgsFieldMappingWidget::removeSelectedFields()
{
  if ( !mTableView->selectionModel()->hasSelection() )
    return false;

  std::list<int> rowsToRemove { selectedRows() };
  rowsToRemove.reverse();
  for ( const int row : rowsToRemove )
  {
    if ( !model()->removeField( model()->index( row, 0, QModelIndex() ) ) )
    {
      return false;
    }
  }
  return true;
}

void QgsFieldMappingWidget::invertSelection()
{
  for ( int i = 0; i < mTableView->model()->rowCount(); ++i )
  {
    for ( int j = 0; j < mTableView->model()->columnCount(); j++ )
    {
      QModelIndex index = mTableView->model()->index( i, j );
      mTableView->selectionModel()->select( index, QItemSelectionModel::Toggle );
    }
  }
}

bool QgsFieldMappingWidget::moveSelectedFieldsUp()
{
  if ( !mTableView->selectionModel()->hasSelection() )
    return false;

  const std::list<int> rowsToMoveUp { selectedRows() };
  for ( const int row : rowsToMoveUp )
  {
    if ( !model()->moveUp( model()->index( row, 0, QModelIndex() ) ) )
    {
      return false;
    }
  }
  return true;
}

bool QgsFieldMappingWidget::moveSelectedFieldsDown()
{
  if ( !mTableView->selectionModel()->hasSelection() )
    return false;

  std::list<int> rowsToMoveDown { selectedRows() };
  rowsToMoveDown.reverse();
  for ( const int row : rowsToMoveDown )
  {
    if ( !model()->moveDown( model()->index( row, 0, QModelIndex() ) ) )
    {
      return false;
    }
  }
  return true;
}

void QgsFieldMappingWidget::updateColumns()
{
  for ( int i = 0; i < mModel->rowCount(); ++i )
  {
    mTableView->openPersistentEditor( mModel->index( i, static_cast<int>( QgsFieldMappingModel::ColumnDataIndex::SourceExpression ) ) );
    mTableView->openPersistentEditor( mModel->index( i, static_cast<int>( QgsFieldMappingModel::ColumnDataIndex::DestinationType ) ) );
  }

  for ( int i = 0; i < mModel->columnCount(); ++i )
  {
    mTableView->resizeColumnToContents( i );
  }
}

std::list<int> QgsFieldMappingWidget::selectedRows()
{
  std::list<int> rows;
  if ( mTableView->selectionModel()->hasSelection() )
  {
    const QModelIndexList constSelection { mTableView->selectionModel()->selectedIndexes() };
    for ( const QModelIndex &index : constSelection )
    {
      rows.push_back( index.row() );
    }
    rows.sort();
    rows.unique();
  }
  return rows;
}

/// @cond PRIVATE

//
// QgsFieldMappingExpressionDelegate
//

QgsFieldMappingExpressionDelegate::QgsFieldMappingExpressionDelegate( QObject *parent )
  : QStyledItemDelegate( parent )
{
}

void QgsFieldMappingExpressionDelegate::setModelData( QWidget *editor, QAbstractItemModel *model, const QModelIndex &index ) const
{
  QgsFieldExpressionWidget *editorWidget { qobject_cast<QgsFieldExpressionWidget *>( editor ) };
  if ( !editorWidget )
    return;

  bool isExpression;
  bool isValid;
  const QString currentValue { editorWidget->currentField( &isExpression, &isValid ) };
  if ( isExpression )
  {
    model->setData( index, currentValue, Qt::EditRole );
  }
  else
  {
    model->setData( index, QgsExpression::quotedColumnRef( currentValue ), Qt::EditRole );
  }
}

void QgsFieldMappingExpressionDelegate::setEditorData( QWidget *editor, const QModelIndex &index ) const
{
  QgsFieldExpressionWidget *editorWidget { qobject_cast<QgsFieldExpressionWidget *>( editor ) };
  if ( !editorWidget )
    return;

  const QVariant value = index.model()->data( index, Qt::EditRole );
  editorWidget->setField( value.toString() );
}

QWidget *QgsFieldMappingExpressionDelegate::createEditor( QWidget *parent, const QStyleOptionViewItem &option, const QModelIndex &index ) const
{
  Q_UNUSED( option )
  QgsFieldExpressionWidget *editor = new QgsFieldExpressionWidget( parent );
  editor->setAutoFillBackground( true );
  editor->setAllowEvalErrors( false );
  if ( const QgsFieldMappingModel *model = qobject_cast<const QgsFieldMappingModel *>( index.model() ) )
  {
    editor->registerExpressionContextGenerator( model->contextGenerator() );
    editor->setFields( model->sourceFields() );
  }
  else if ( const QgsAggregateMappingModel *model = qobject_cast<const QgsAggregateMappingModel *>( index.model() ) )
  {
    editor->registerExpressionContextGenerator( model->contextGenerator() );
    editor->setFields( model->sourceFields() );
  }
  else
  {
    Q_ASSERT( false );
  }

  if ( QgsFieldMappingWidget *mappingWidget = qobject_cast<QgsFieldMappingWidget *>( QgsFieldMappingExpressionDelegate::parent() ) )
  {
    if ( mappingWidget->sourceLayer() )
      editor->setLayer( mappingWidget->sourceLayer() );
  }
  else if ( QgsAggregateMappingWidget *aggregateWidget = qobject_cast<QgsAggregateMappingWidget *>( QgsFieldMappingExpressionDelegate::parent() ) )
  {
    if ( aggregateWidget->sourceLayer() )
      editor->setLayer( aggregateWidget->sourceLayer() );
  }

  editor->setField( index.model()->data( index, Qt::DisplayRole ).toString() );
  connect( editor, qOverload<const QString &>( &QgsFieldExpressionWidget::fieldChanged ), this, [this, editor]( const QString &fieldName ) {
    Q_UNUSED( fieldName )
    const_cast<QgsFieldMappingExpressionDelegate *>( this )->emit commitData( editor );
  } );
  return editor;
}


//
// QgsFieldMappingTypeDelegate
//

QgsFieldMappingTypeDelegate::QgsFieldMappingTypeDelegate( const QList< QgsVectorDataProvider::NativeType > &nativeTypes, QObject *parent )
  : QStyledItemDelegate( parent )
  , mNativeTypes( nativeTypes.isEmpty() ? QgsFieldMappingModel::supportedDataTypes() : nativeTypes )
{
}

QWidget *QgsFieldMappingTypeDelegate::createEditor( QWidget *parent, const QStyleOptionViewItem &option, const QModelIndex &index ) const
{
  Q_UNUSED( option )
  QComboBox *editor = new QComboBox( parent );

  int i = 0;
  for ( const QgsVectorDataProvider::NativeType &type : mNativeTypes )
  {
    editor->addItem( QgsFields::iconForFieldType( type.mType, type.mSubType, type.mTypeName ), type.mTypeDesc );
    editor->setItemData( i, type.mTypeName, Qt::UserRole );
    i++;
  }

  const QgsFieldMappingModel *model { qobject_cast<const QgsFieldMappingModel *>( index.model() ) };

  if ( model && !model->destinationEditable() )
  {
    editor->setEnabled( false );
  }
  else
  {
    connect( editor, qOverload<int>( &QComboBox::currentIndexChanged ), this, [this, editor]( int currentIndex ) {
      Q_UNUSED( currentIndex )
      const_cast<QgsFieldMappingTypeDelegate *>( this )->emit commitData( editor );
    } );
  }
  return editor;
}

void QgsFieldMappingTypeDelegate::setEditorData( QWidget *editor, const QModelIndex &index ) const
{
  QComboBox *editorWidget { qobject_cast<QComboBox *>( editor ) };
  if ( !editorWidget )
    return;

  const QVariant value = index.model()->data( index, Qt::EditRole );
  editorWidget->setCurrentIndex( editorWidget->findData( value ) );
}

void QgsFieldMappingTypeDelegate::setModelData( QWidget *editor, QAbstractItemModel *model, const QModelIndex &index ) const
{
  QComboBox *editorWidget { qobject_cast<QComboBox *>( editor ) };
  if ( !editorWidget )
    return;

  const QVariant currentValue = editorWidget->currentData();
  model->setData( index, currentValue, Qt::EditRole );
}

void QgsFieldMappingTypeDelegate::setNativeTypes( const QList<QgsVectorDataProvider::NativeType> &nativeTypes )
{
  mNativeTypes = nativeTypes.isEmpty() ? QgsFieldMappingModel::supportedDataTypes() : nativeTypes;
}

/// @endcond
