/***************************************************************************
         qgsscalecombobox.h
         ------------------------
  begin                : January 7, 2012
  copyright            : (C) 2012 by Alexander Bruy
  email                : alexander dot bruy at gmail dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qgsscalecombobox.h"

#include "qgis.h"
#include "qgsmathutils.h"
#include "qgssettingsentryimpl.h"
#include "qgssettingsregistrycore.h"

#include <QAbstractItemView>
#include <QLineEdit>
#include <QLocale>

#include "moc_qgsscalecombobox.cpp"

QgsScaleComboBox::QgsScaleComboBox( QWidget *parent )
  : QComboBox( parent )
{
  updateScales();

  setEditable( true );
  setInsertPolicy( QComboBox::NoInsert );
  setCompleter( nullptr );
  connect( this, qOverload<int>( &QComboBox::activated ), this, &QgsScaleComboBox::fixupScale );
  connect( lineEdit(), &QLineEdit::editingFinished, this, &QgsScaleComboBox::fixupScale );
  fixupScale();
}

void QgsScaleComboBox::updateScales( const QStringList &scales )
{
  QStringList scalesList;
  const QString oldScale = currentText();

  if ( scales.isEmpty() )
  {
    scalesList = QgsSettingsRegistryCore::settingsMapScales->value();
  }
  else
  {
    scalesList = scales;
  }

  QStringList cleanedScalesList;
  for ( const QString &scale : std::as_const( scalesList ) )
  {
    const QStringList parts = scale.split( ':' );
    if ( parts.size() < 2 )
      continue;

    bool ok = false;
    const double denominator = QLocale().toDouble( parts[1], &ok );
    if ( ok )
    {
      cleanedScalesList.push_back( toString( denominator ) );
    }
    else
    {
      const double denominator = parts[1].toDouble( &ok );
      if ( ok )
      {
        cleanedScalesList.push_back( toString( denominator ) );
      }
    }
  }

  blockSignals( true );
  clear();
  addItems( cleanedScalesList );
  setScaleString( oldScale );
  blockSignals( false );
}

void QgsScaleComboBox::setPredefinedScales( const QVector<double> &scales )
{
  if ( scales.isEmpty() )
  {
    updateScales();
    return;
  }

  const QString oldScale = currentText();

  QStringList scalesStringList;
  scalesStringList.reserve( scales.size() );
  for ( double denominator : scales )
  {
    scalesStringList.push_back( toString( denominator ) );
  }

  blockSignals( true );
  clear();
  addItems( scalesStringList );
  setScaleString( oldScale );
  blockSignals( false );
}

void QgsScaleComboBox::showPopup()
{
  QComboBox::showPopup();

  if ( !currentText().contains( ':' ) )
  {
    return;
  }
  QStringList parts = currentText().split( ':' );
  bool ok;
  int idx = 0;
  int min = 999999;
  const long currScale = parts.at( 1 ).toLong( &ok );
  long nextScale, delta;
  for ( int i = 0; i < count(); i++ )
  {
    parts = itemText( i ).split( ':' );
    nextScale = parts.at( 1 ).toLong( &ok );
    delta = std::labs( currScale - nextScale );
    if ( delta < min )
    {
      min = delta;
      idx = i;
    }
  }

  blockSignals( true );
  view()->setCurrentIndex( model()->index( idx, 0 ) );
  blockSignals( false );
  view()->setMinimumWidth( view()->sizeHintForColumn( 0 ) );
}

QString QgsScaleComboBox::scaleString() const
{
  return toString( mScale, mMode );
}

bool QgsScaleComboBox::setScaleString( const QString &string )
{
  const double oldScale = mScale;
  if ( mAllowNull && string.trimmed().isEmpty() )
  {
    mScale = std::numeric_limits<double>::quiet_NaN();
    setEditText( toString( mScale ) );
    clearFocus();
    if ( !std::isnan( oldScale ) )
    {
      emit scaleChanged( mScale );
    }
    return true;
  }

  bool ok;
  double newScale = toDouble( string, &ok );
  if ( newScale > mMinScale && newScale != 0 && mMinScale != 0 )
  {
    newScale = mMinScale;
  }
  if ( !ok )
  {
    return false;
  }
  else
  {
    mScale = newScale;
    setEditText( toString( mScale, mMode ) );
    clearFocus();
    if ( mScale != oldScale )
    {
      emit scaleChanged( mScale );
    }
    return true;
  }
}

double QgsScaleComboBox::scale() const
{
  return mScale;
}

bool QgsScaleComboBox::isNull() const
{
  return std::isnan( mScale );
}

void QgsScaleComboBox::setScale( double scale )
{
  setScaleString( toString( scale, mMode ) );
}

void QgsScaleComboBox::fixupScale()
{
  if ( mAllowNull && currentText().trimmed().isEmpty() )
  {
    setScale( std::numeric_limits<double>::quiet_NaN() );
    return;
  }

  const QStringList txtList = currentText().split( ':' );
  const bool userSetScale = txtList.size() != 2;

  bool ok;
  double newScale = toDouble( currentText(), &ok );

  // Valid string representation
  if ( ok )
  {
    switch ( mMode )
    {
      case RatioMode::ForceUnitNumerator:
      {
        // if a user types scale = 2345, we transform to 1:2345
        if ( userSetScale && newScale < 1.0 && !qgsDoubleNear( newScale, 0.0 ) )
        {
          newScale = 1 / newScale;
        }
        break;
      }
      case RatioMode::Flexible:
        break;
    }

    setScale( newScale );
  }
  else
  {
    setScale( mScale );
  }
}

QgsScaleComboBox::RatioMode QgsScaleComboBox::ratioMode() const
{
  return mMode;
}

void QgsScaleComboBox::setRatioMode( QgsScaleComboBox::RatioMode mode )
{
  if ( mode == mMode )
    return;

  mMode = mode;
  setScale( mScale );
  emit ratioModeChanged( mMode );
}

QString QgsScaleComboBox::toString( double scale, RatioMode mode )
{
  if ( std::isnan( scale ) )
  {
    return QString();
  }
  if ( scale == 0 )
  {
    return QStringLiteral( "0" );
  }

  switch ( mode )
  {
    case RatioMode::ForceUnitNumerator:
      if ( scale <= 1 )
      {
        return QStringLiteral( "%1:1" ).arg( QLocale().toString( static_cast<int>( std::round( 1.0 / scale ) ) ) );
      }
      else
      {
        return QStringLiteral( "1:%1" ).arg( QLocale().toString( static_cast<float>( std::round( scale ) ), 'f', 0 ) );
      }

    case RatioMode::Flexible:
    {
      qlonglong numerator = 0;
      qlonglong denominator = 0;
      QgsMathUtils::doubleToRational( 1.0 / scale, numerator, denominator, 0.01 );
      return QStringLiteral( "%1:%2" ).arg(
        QLocale().toString( numerator ),
        QLocale().toString( denominator )
      );
    }
  }
  return QString();
}

double QgsScaleComboBox::toDouble( const QString &scaleString, bool *returnOk )
{
  bool ok = false;
  QString scaleTxt( scaleString );

  const double denominator = qgsPermissiveToDouble( scaleTxt, ok );
  double scale = !qgsDoubleNear( denominator, 0.0 ) ? 1.0 / denominator : 0.0;
  if ( ok )
  {
    // Create a text version and set that text and rescan
    // Idea is to get the same rounding.
    scaleTxt = toString( scale );
  }
  else
  {
    // It is now either X:Y or not valid
    QStringList txtList = scaleTxt.split( ':' );
    if ( 2 == txtList.size() )
    {
      bool okX = false;
      bool okY = false;
      const int x = qgsPermissiveToInt( txtList[0], okX );
      const int y = qgsPermissiveToInt( txtList[1], okY );
      if ( okX && okY && x != 0 )
      {
        // Scale is fraction of x and y
        scale = static_cast<double>( y ) / static_cast<double>( x );
        ok = true;
      }
    }
  }

  // Set up optional return flag
  if ( returnOk )
  {
    *returnOk = ok;
  }
  return scale;
}

void QgsScaleComboBox::setAllowNull( bool allowNull )
{
  mAllowNull = allowNull;
  lineEdit()->setClearButtonEnabled( allowNull );
  updateScales();
}

bool QgsScaleComboBox::allowNull() const
{
  return mAllowNull;
}

void QgsScaleComboBox::setMinScale( double scale )
{
  mMinScale = scale;
  if ( mScale > mMinScale && mScale != 0 && mMinScale != 0 )
  {
    setScale( mMinScale );
  }
}

void QgsScaleComboBox::setNull()
{
  if ( allowNull() )
    setScale( std::numeric_limits<double>::quiet_NaN() );
}
